C      *****************************************************************
C      * Function Davies                                               *
C      * Returns the Davies activity coefficient given ionic strength  *
C      * and ionic charge for a dissolved species.                     *
C      *                                                               *
C      * Variables:                                                    *
C      * DaviesA   -Local - Constant used in Davies equation.  The     *
C      *                    equation in which these constants are used *
C      *                    is not going to be made more or less       *
C      *                    comprehensible by the units.               *
C      * DaviesB   -Local - Constant used in Davies equation.  The     *
C      *                    equation in which these constants are used *
C      *                    is not going to be made more or less       *
C      *                    comprehensible by the units.               *
C      * IStr      -Input - Ionic strength.  Units are mol/L.  (These  *
C      *                    units do not result from the calculation   *
C      *                    but are defined by usage (Morel 1983).)    *
C      * SpChg     -Input - Species charge.  Units are equivalent      *
C      *                    charge.                                    *
C      * T1, T2    -Local - Temporary storage.  Units are consistent   *
C      *                    with items stored.                         *
C      *****************************************************************
       REAL*8 FUNCTION DAVIES(ISTR, SPCHG)
							IMPLICIT NONE
       REAL*8 DAVIESA, DAVIESB, ISTR, T1, T2
       INTEGER SPCHG

C      *-----------------------------------------------------*
C      * Davies coefficient is undefined for neutral species *
C      *-----------------------------------------------------*
       IF (SPCHG.EQ.0) THEN
          DAVIES = 1.0
          GOTO 9999
       ENDIF

C      *--------------------------*
C      * Set up Davies constants. *
C      *--------------------------*
       DAVIESA = 0.5
       DAVIESB = 0.2

C      *-----------------------------------*
C      * Calculate the Davies coefficient. *
C      *-----------------------------------*
       T1 = SQRT(ISTR)
       T1 = T1 / (1 + T1)
       T2 = -DAVIESA * (SPCHG**2) * (T1 - (DAVIESB * ISTR))

C      *--------------------------------------------------------*
C      * To avoid overflows ignore values of T2 greater than 3. *
C      * It is expected that T2 is generally less than 1.       *
C      *--------------------------------------------------------*
       IF (T2.GT.3.0) THEN
C         *------------------------------------------------------------*
C         * Numbers this large generally result from wild fluctuations *
C         * in component concentrations due to the iterative nature of *
C         * the numerical solution routine.  They will go away.        *
C         *------------------------------------------------------------*
          DAVIES = 1.0
       ELSE
          DAVIES = 10.0 ** T2
       ENDIF

C      *--------------*
C      * Escape hatch *
C      *--------------*
 9999  CONTINUE

	      RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
